<?php

/**
 * This is the model class for table "Uzytkownik".
 *
 * The followings are the available columns in table 'Uzytkownik':
 * @property double $id
 * @property string $email
 * @property string $haslo
 * @property string $pesel
 * @property string $imie
 * @property string $nazwisko
 * @property double $idFirmy
 * @property double $idKlasy
 * @property string $plec
 * @property string $kodDoOdzyskaniaHasla
 * @property double $status
 */
class Uzytkownik extends CActiveRecord
{
	const USER_STATUS_INACTIVE = 0;
	const USER_STATUS_ACTIVE = 1;
	const USER_STATUS_DELETED = 2;
	const USER_STATUS_EXCLUDED = 3;

	public $className;
	public $companyName;
	public $schoolName;
	public $schoolId;
	public $classId;
	public $hasloPowtorzone;
	public $rulesAccepted;

	public $restrictedEmails = array();
	public $restrictedPesels = array();

	/**
	 * Returns the static model of the specified AR class.
	 * @param string $className active record class name.
	 * @return Uzytkownik the static model class
	 */
	public static function model($className=__CLASS__)
	{
		return parent::model($className);
	}

	/**
	 * @return string the associated database table name
	 */
	public function tableName()
	{
		return 'Uzytkownik';
	}

	public function emailNotRestricted()
	{
		$occurences = array_count_values($this->restrictedEmails);
		if(isset($occurences[$this->email]) && $occurences[$this->email]>1)
			$this->addError('email','email "'.$this->email.'" został już wykorzystany.');
	}

	public function peselNotRestricted()
	{
		$occurences = array_count_values($this->restrictedPesels);
		if(isset($occurences[$this->pesel]) && $occurences[$this->pesel]>1)
			$this->addError('pesel','PESEL "'.$this->pesel.'" został już wykorzystany.');
	}

	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		return array(
			array('className,schoolName,companyName,schoolId,classId,idKlasy,status','safe'),
			array('imie,nazwisko,email,email,haslo,hasloPowtorzone,plec', 'required','on'=>'register'),
			array('rulesAccepted', 'compare', 'compareValue' => true, 'message' => 'Musisz zaakceptować regulamin.','on'=>'register'),
			array('hasloPowtorzone', 'compare', 'compareAttribute'=>'haslo', 'message'=>'Hasło musi być powtórzone dokładnie','on'=>'register'),
			array('email','emailNotRestricted','on'=>'register'),
			array('pesel', 'ext.Validators.PeselValidator', 'disableChecksum' => ParametryGry::getValue('CZY_TRYB_DEMO', 0)),
			array('idFirmy, idKlasy', 'numerical'),
			array('email', 'length', 'max'=>508),
			array('email', 'email', 'on'=>'register'),
			array('haslo, imie, nazwisko', 'length', 'max'=>128),
			array('pesel', 'length', 'is'=>11),
			array('pesel', 'numerical'),
			array('plec', 'length', 'max'=>1),
			array('email', 'unique'),
			array('kodDoOdzyskaniaHasla', 'length', 'max'=>64),

			array('id, email, haslo, pesel, imie, nazwisko, idFirmy, idKlasy, plec', 'safe', 'on'=>'search'),
		);
	}

	/**
	 * @return array relational rules.
	 */
	public function relations()
	{
		return array(
			'dziedzinaUzytkownikas'=>array(self::HAS_MANY, 'DziedzinaUzytkownika', 'idUzytkownika'),
			'firma'=>array(self::BELONGS_TO, 'Firma', 'idFirmy'),
			'klasa'=>array(self::BELONGS_TO, 'Klasa', 'idKlasy'),
			'formularze'=>array(self::HAS_MANY, 'FormularzUzytkownika', 'idUzytkownika'),
			'formularzeWypelnione'=>array(self::HAS_MANY, 'FormularzUzytkownika', 'idUzytkownika', 'condition'=>'"formularzeWypelnione"."status" IN('.FormularzUzytkownika::STATUS_FILLED.', '.FormularzUzytkownika::STATUS_ACCEPTED.', '.FormularzUzytkownika::STATUS_REFUSED.')'),
			'zlecenia'=>array(self::HAS_MANY, 'ZlecenieFirmy', 'idUcznia')
		);
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'id'=>Yii::t('Uzytkownik', 'ID'),
			'email'=>Yii::t('Uzytkownik', 'E-mail'),
			'haslo'=>Yii::t('Uzytkownik', 'Hasło'),
			'PESEL'=>Yii::t('Uzytkownik', 'PESEL'),
			'imie'=>Yii::t('Uzytkownik', 'Imię'),
			'nazwisko'=>Yii::t('Uzytkownik', 'Nazwisko'),
			'idFirmy'=>Yii::t('Uzytkownik', 'Firma'),
			'idKlasy'=>Yii::t('Uzytkownik', 'Klasa'),
			'plec'=>Yii::t('Uzytkownik', 'Płeć'),
		);
	}

	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 * @return CActiveDataProvider the data provider that can return the models based on the search/filter conditions.
	 */
	public function search($param = array())
	{
		$criteria = new CDbCriteria($param);

		$classes = Yii::app()->user->getState('classes');
		if(Yii::app()->user->role == 'schoolAdmin' && !empty($classes))
			$criteria->addInCondition(Yii::app()->db->quoteTableName("t").'.'.Yii::app()->db->quoteColumnName("idKlasy"), Yii::app()->user->getState('classes'));
		elseif(Yii::app()->user->role == 'schoolAdmin' && empty($classes))
			return new CActiveDataProvider($this, array('data'=>array()));

		$criteria->compare(Yii::app()->db->quoteTableName("t").'.'.Yii::app()->db->quoteColumnName("id"), $this->id);
		$criteria->compare('LOWER('.Yii::app()->db->quoteColumnName("email").')', strtolower($this->email), true);
		$criteria->compare(Yii::app()->db->quoteColumnName("haslo"), $this->haslo, true);
		$criteria->compare(Yii::app()->db->quoteColumnName("pesel"), $this->pesel, true);
		$criteria->compare('LOWER('.Yii::app()->db->quoteColumnName("imie").')', strtolower($this->imie), true);
		$criteria->compare('LOWER('.Yii::app()->db->quoteColumnName("nazwisko").')', strtolower($this->nazwisko), true);
		$criteria->compare(Yii::app()->db->quoteColumnName("idFirmy"), $this->idFirmy);
		$criteria->compare(Yii::app()->db->quoteTableName("t").'.'.Yii::app()->db->quoteColumnName("idKlasy"), $this->idKlasy);
		$criteria->compare(Yii::app()->db->quoteColumnName("plec"), $this->plec, true);
		$criteria->compare('LOWER('.Yii::app()->db->quoteTableName("klasa").'.'.Yii::app()->db->quoteColumnName("nazwa").')',strtolower($this->className),true);
		$criteria->compare('LOWER('.Yii::app()->db->quoteTableName("klasa").'.'.Yii::app()->db->quoteTableName("szkola").'.'.Yii::app()->db->quoteColumnName("nazwa").')',strtolower($this->schoolName),true);
		$criteria->compare('LOWER('.Yii::app()->db->quoteTableName("firma").'.'.Yii::app()->db->quoteColumnName("nazwa").')',strtolower($this->companyName),true);

		$criteria->with = array('firma','klasa','klasa.szkola');

		return new CActiveDataProvider($this, array(
			'criteria'=>$criteria,
		));
	}

	public function validatePassword($password)
	{
		return crypt($password,$this->haslo)===$this->haslo;
	}

	public function hashPassword($password)
	{
		return crypt($password, $this->generateSalt());
	}

	public function hasEmptyRegisterAttributes()
	{
		return empty($this->imie) && empty($this->nazwisko) && empty($this->pesel) && empty($this->email) && empty($this->haslo) && empty($this->hasloPowtorzone);
	}

	public function register()
	{
		if($this->hasEmptyRegisterAttributes())
			return;

		$password = $this->haslo;
		$this->haslo = ' ';
		$this->dataRejestracji = new CDbExpression('SYSDATE');

		$this->save(false);

		$this->haslo = hash('sha1', $password);

		$this->save(false);

		$fields = Dziedzina::model()->findAll();

		foreach($fields as $field)
		{
			$userField = new DziedzinaUzytkownika();
                       /* $userField->idUzytkownika = $this->id;
			$userField->idDziedziny = $field->id;*/
			$userField->idUzytkownika = intval($this->id);
			$userField->idDziedziny = intval($field->id);
			$userField->punkty = intval($userField->punkty);
                        /*echo var_dump($userField->idUzytkownika).'<br>';
                        echo var_dump($userField->idDziedziny).'<br>';
                        echo var_dump($userField->punkty).'<br>';
                        die();*/
                       // echo $userField->getCommandBuilder()->createInsertCommand($userField->getMetaData()->tableSchema, $userField->getAttributes())->getText();die();
                        //var_dump($userField);die();
			$userField->save(false);
		}
	}

	public function getAllStatistics($classId)
	{
		$sql = 'SELECT t5."imie", t5."nazwisko", "punkty", t."idUzytkownika" as "id", t."idUzytkownika", t5."idFirmy"
			FROM "PunktyUzytkownika" t
			LEFT JOIN "Uzytkownik" t5 ON t5."id" = t."idUzytkownika"
			WHERE t5."idKlasy" = '.Yii::app()->db->quoteValue($classId).'
			ORDER BY "punkty" DESC';

			return new CSqlDataProvider($sql,array(
				'pagination'=>false
			));
	}

	public function getAllStatisticsTotal()
	{
		$sql = 'SELECT t."idUzytkownika" as "id", t5."imie", t5."nazwisko", "punkty", k."nazwa" AS "klasa", s."nazwa" AS "szkola"
		FROM "PunktyUzytkownika" t
		LEFT JOIN "Uzytkownik" t5 ON t5."id" = t."idUzytkownika"
		LEFT JOIN "Klasa" k ON k."id" = t5."idKlasy"
		LEFT JOIN "Szkola" s ON s."id" = k."idSzkoly"
		';

		$count = Yii::app()->db->createCommand('SELECT COUNT(*) FROM "PunktyUzytkownika"')->queryScalar();

		$schoolid = Yii::app()->user->getState('schoolId');
		$classese = Yii::app()->user->getState('classes');
		if(Yii::app()->user->role == 'schoolAdmin' && !empty($schoolid) && !empty($classese))
		{
			$sql = 'SELECT t."idUzytkownika" as "id", t5."imie", t5."nazwisko", "punkty", k."nazwa" AS "klasa", s."nazwa" AS "szkola"
			FROM "PunktyUzytkownika" t
			LEFT JOIN "Uzytkownik" t5 ON t5."id" = t."idUzytkownika"
			LEFT JOIN "Klasa" k ON k."id" = t5."idKlasy"
			LEFT JOIN "Szkola" s ON s."id" = k."idSzkoly"
			WHERE s."id" = '.Yii::app()->user->getState('schoolId').'
			';

			$count = Yii::app()->db->createCommand('select count(*) from "PunktyUzytkownika" p INNER JOIN "Uzytkownik" u on u."id" = p."idUzytkownika" WHERE u."idKlasy" IN ('.implode(',', Yii::app()->user->getState('classes')).')')->queryScalar();
		}

        // return new CSqlDataProvider($sql,array(
        //         'pagination'=>false
        // ));
		return new CSqlDataProvider($sql, array(
			'totalItemCount'=>$count,
			'pagination'=>array(
				'pageSize'=>100,
				)
			));
	}

	public function getStatistics()
	{
		$sql = 'SELECT t3."idUzytkownika" as "id", t."nazwa", NVL(t3."punkty", 0) as "punkty"
				FROM "Dziedzina" t
				LEFT JOIN (SELECT *
						FROM "PunktyDziedzinyUzytkownika" t2
						WHERE "idUzytkownika" = '.$this->id.') t3 ON t."id" = t3."idDziedziny"
				ORDER BY "punkty" DESC';

		return new CSqlDataProvider($sql,array(
			'pagination'=>false
		));
	}

	public function getZleceniaStatistics()
	{
		$pobraneSql = 'SELECT "nazwa", SUM("pobrane") AS "pobrane" FROM (
				SELECT d."nazwa" as "nazwa", COUNT(zf."id") AS "pobrane" FROM "ZlecenieFirmy" zf
				LEFT JOIN "Zlecenie" z ON z."id" = zf."idZlecenia"
				LEFT JOIN "Dziedzina" d ON d."id" = z."idDziedziny"
				WHERE zf."idUcznia" = :userId
				GROUP BY d."nazwa"
			UNION
				SELECT d."nazwa", 0 FROM "Dziedzina" d
			) GROUP BY "nazwa" ORDER BY 2 DESC';

		$rozwiazaneSql = 'SELECT "nazwa", SUM("rozwiazane") AS "rozwiazane" FROM (
				SELECT d."nazwa" as "nazwa", COUNT(zf."id") AS "rozwiazane" FROM "ZlecenieFirmy" zf
				LEFT JOIN "Zlecenie" z ON z."id" = zf."idZlecenia"
				LEFT JOIN "Dziedzina" d ON d."id" = z."idDziedziny"
				WHERE zf."idUcznia" = :userId AND zf."idOdpowiedzi" IS NOT NULL
				GROUP BY d."nazwa"
			UNION
				SELECT d."nazwa", 0 FROM "Dziedzina" d
			) GROUP BY "nazwa" ORDER BY 2 DESC';

		$pobrane = Yii::app()->db->createCommand($pobraneSql)->queryAll(true, array(':userId'=>$this->id));
		$rozwiazane = Yii::app()->db->createCommand($rozwiazaneSql)->queryAll(true, array(':userId'=>$this->id));

		$resultArray = array();
		foreach($pobrane as $i=>$pobr)
		{
			$resultArray[$i]['nazwa'] = $pobr['nazwa'];
			$resultArray[$i]['pobrane'] = $pobr['pobrane'];
		}

		foreach($rozwiazane as $i=>$rozw)
			$resultArray[$i]['rozwiazane'] = $rozw['rozwiazane'];

		return new CArrayDataProvider($resultArray,array(
			'keyField'=>'nazwa',
			'pagination'=>false
		));
	}

	public static function getTeachersStudent($studentId)
	{
		return Uzytkownik::model()->find(array(
			'join' => 'LEFT JOIN "AdminKlasy" "ak" ON "t"."idKlasy" = "ak"."idKlasy"',
			'condition' => '"t"."id" = :studentId AND "ak"."idAdmina" = :teacherId',
			'params' => array(
				':studentId' => $studentId,
				':teacherId' => Yii::app()->user->id
			)
		));
	}

	public function getImieNazwisko()
	{
		return $this->imie . ' ' . $this->nazwisko;
	}

	public static function isCompanyOwner($userId, $companyId)
	{
		return (int)Yii::app()->db->createCommand('
			select count(*) from "Uzytkownik" u
			where u."id" = :userId and u."status" = 1 and u."idFirmy" = :companyId
		')->queryScalar(array(
			':companyId' => $companyId,
			':userId' => $userId,
		));
	}
}
